<?php
/**
 * Bank List Comparison Test
 * 
 * This script compares API bank list with hardcoded bank list
 * and shows which one is currently being used
 */

header('Content-Type: text/html; charset=utf-8');

require_once 'bank_list.php';

// Load API configuration
$configFile = __DIR__ . '/api_config.php';
if (file_exists($configFile)) {
    $config = require $configFile;
    $credentials = $config['credentials'] ?? [];
    $defaultMerchantId = $credentials['merchant_id'] ?? $credentials['super_merchant_id'] ?? null;
    $certPath = $config['certificate_path'] ?? (__DIR__ . '/../fingpay_public_production.txt');
} else {
    $defaultMerchantId = null;
    $certPath = __DIR__ . '/../fingpay_public_production.txt';
}

?>
<!DOCTYPE html>
<html>
<head>
    <title>Bank List Comparison</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 20px;
            background: #f5f5f5;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #333;
        }
        .status {
            padding: 15px;
            margin: 20px 0;
            border-radius: 5px;
            font-weight: bold;
        }
        .hardcoded { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .api-success { background: #d1ecf1; color: #0c5460; border: 1px solid #bee5eb; }
        .api-failed { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .comparison {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
            margin: 20px 0;
        }
        .bank-list {
            border: 1px solid #ddd;
            padding: 15px;
            border-radius: 5px;
            max-height: 500px;
            overflow-y: auto;
        }
        .bank-item {
            padding: 8px;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
        }
        .match { background: #d4edda; }
        .diff { background: #fff3cd; }
        .missing { background: #f8d7da; }
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
        }
        th, td {
            padding: 10px;
            text-align: left;
            border: 1px solid #ddd;
        }
        th {
            background: #667eea;
            color: white;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🏦 Bank List Comparison Report</h1>
        
        <?php
        // Get current setting
        $currentUseAPI = false; // From get_bank_list.php
        
        // Get Hardcoded List
        echo "<h2>1. Current Status</h2>";
        echo "<div class='status hardcoded'>";
        echo "✅ Currently using: <strong>HARDCODED LIST</strong><br>";
        echo "   (API is disabled: \$useAPI = false in get_bank_list.php)";
        echo "</div>";
        
        // Get Hardcoded List
        echo "<h2>2. Hardcoded Bank List</h2>";
        $hardcodedList = getBankList($certPath, $defaultMerchantId, false);
        $hardcodedCount = count($hardcodedList);
        echo "<div class='status hardcoded'>";
        echo "Total Hardcoded Banks: <strong>$hardcodedCount</strong>";
        echo "</div>";
        
        // Try to get API List
        echo "<h2>3. API Bank List (Testing)</h2>";
        $apiList = [];
        $apiSuccess = false;
        $apiError = '';
        
        try {
            $apiList = getBankList($certPath, $defaultMerchantId, true);
            if (!empty($apiList)) {
                $apiSuccess = true;
                $apiCount = count($apiList);
                echo "<div class='status api-success'>";
                echo "✅ API Success: <strong>$apiCount banks</strong> fetched from API";
                echo "</div>";
            } else {
                $apiError = "API returned empty list or failed";
                echo "<div class='status api-failed'>";
                echo "❌ API Failed: $apiError (falling back to hardcoded list)";
                echo "</div>";
            }
        } catch (Exception $e) {
            $apiError = $e->getMessage();
            echo "<div class='status api-failed'>";
            echo "❌ API Error: " . htmlspecialchars($apiError);
            echo "</div>";
        }
        
        // Comparison
        echo "<h2>4. Comparison Results</h2>";
        
        if ($apiSuccess && !empty($apiList)) {
            // Both lists available - compare them
            $allCodes = array_unique(array_merge(array_keys($hardcodedList), array_keys($apiList)));
            $matches = 0;
            $differences = [];
            $onlyHardcoded = [];
            $onlyAPI = [];
            
            foreach ($allCodes as $code) {
                $hardcodedName = $hardcodedList[$code] ?? null;
                $apiName = $apiList[$code] ?? null;
                
                if ($hardcodedName && $apiName) {
                    if ($hardcodedName === $apiName) {
                        $matches++;
                    } else {
                        $differences[] = [
                            'code' => $code,
                            'hardcoded' => $hardcodedName,
                            'api' => $apiName
                        ];
                    }
                } elseif ($hardcodedName && !$apiName) {
                    $onlyHardcoded[] = ['code' => $code, 'name' => $hardcodedName];
                } elseif (!$hardcodedName && $apiName) {
                    $onlyAPI[] = ['code' => $code, 'name' => $apiName];
                }
            }
            
            echo "<table>";
            echo "<tr><th>Metric</th><th>Count</th></tr>";
            echo "<tr><td>Total Banks in Hardcoded</td><td>$hardcodedCount</td></tr>";
            echo "<tr><td>Total Banks in API</td><td>" . count($apiList) . "</td></tr>";
            echo "<tr class='match'><td>✅ Exact Matches</td><td>$matches</td></tr>";
            echo "<tr class='diff'><td>⚠️ Same Code, Different Names</td><td>" . count($differences) . "</td></tr>";
            echo "<tr class='missing'><td>❌ Only in Hardcoded</td><td>" . count($onlyHardcoded) . "</td></tr>";
            echo "<tr class='missing'><td>➕ Only in API</td><td>" . count($onlyAPI) . "</td></tr>";
            echo "</table>";
            
            // Check if lists are same
            $areSame = (count($differences) == 0 && count($onlyHardcoded) == 0 && count($onlyAPI) == 0 && $matches == $hardcodedCount);
            
            if ($areSame) {
                echo "<div class='status hardcoded' style='font-size: 18px; padding: 20px; text-align: center;'>";
                echo "✅ <strong>PERFECT MATCH!</strong><br>";
                echo "Hardcoded list and API list are <strong>IDENTICAL</strong>.<br>";
                echo "All $matches banks match perfectly!";
                echo "</div>";
            } else {
                echo "<div class='status api-failed'>";
                echo "⚠️ Lists are <strong>NOT IDENTICAL</strong><br>";
                echo "There are differences between hardcoded and API lists.";
                echo "</div>";
                
                // Show differences
                if (!empty($differences)) {
                    echo "<h3>Banks with Different Names:</h3>";
                    echo "<table>";
                    echo "<tr><th>Bank Code</th><th>Hardcoded Name</th><th>API Name</th></tr>";
                    foreach ($differences as $diff) {
                        echo "<tr>";
                        echo "<td>" . htmlspecialchars($diff['code']) . "</td>";
                        echo "<td>" . htmlspecialchars($diff['hardcoded']) . "</td>";
                        echo "<td>" . htmlspecialchars($diff['api']) . "</td>";
                        echo "</tr>";
                    }
                    echo "</table>";
                }
                
                if (!empty($onlyHardcoded)) {
                    echo "<h3>Banks Only in Hardcoded List:</h3>";
                    echo "<ul>";
                    foreach (array_slice($onlyHardcoded, 0, 10) as $bank) {
                        echo "<li>" . htmlspecialchars($bank['code']) . " - " . htmlspecialchars($bank['name']) . "</li>";
                    }
                    if (count($onlyHardcoded) > 10) {
                        echo "<li>... and " . (count($onlyHardcoded) - 10) . " more</li>";
                    }
                    echo "</ul>";
                }
                
                if (!empty($onlyAPI)) {
                    echo "<h3>Banks Only in API List:</h3>";
                    echo "<ul>";
                    foreach (array_slice($onlyAPI, 0, 10) as $bank) {
                        echo "<li>" . htmlspecialchars($bank['code']) . " - " . htmlspecialchars($bank['name']) . "</li>";
                    }
                    if (count($onlyAPI) > 10) {
                        echo "<li>... and " . (count($onlyAPI) - 10) . " more</li>";
                    }
                    echo "</ul>";
                }
            }
            
        } else {
            // API not available
            echo "<div class='status api-failed'>";
            echo "❌ Cannot compare: API is not available or returned error<br>";
            echo "Currently using <strong>HARDCODED LIST</strong> with <strong>$hardcodedCount banks</strong>";
            echo "</div>";
            
            echo "<h3>Hardcoded Bank List:</h3>";
            echo "<div class='bank-list'>";
            foreach ($hardcodedList as $code => $name) {
                echo "<div class='bank-item'>";
                echo "<span><strong>" . htmlspecialchars($code) . "</strong></span>";
                echo "<span>" . htmlspecialchars($name) . "</span>";
                echo "</div>";
            }
            echo "</div>";
        }
        
        // Summary
        echo "<h2>5. Summary & Recommendation</h2>";
        echo "<div class='status hardcoded'>";
        if ($apiSuccess && $areSame) {
            echo "<strong>✅ PERFECT!</strong><br>";
            echo "API and hardcoded lists are identical. You can safely:<br>";
            echo "• Keep using hardcoded list (faster, no API dependency)<br>";
            echo "• Or enable API (will return same data)<br>";
            echo "<br>Both options are perfect!";
        } elseif ($apiSuccess && !$areSame) {
            echo "<strong>⚠️ LISTS DIFFER</strong><br>";
            echo "API and hardcoded lists are different.<br>";
            echo "• Hardcoded has $hardcodedCount banks<br>";
            echo "• API has " . count($apiList) . " banks<br>";
            echo "<br>Recommendation: Review and update hardcoded list if API data is more accurate.";
        } else {
            echo "<strong>📋 CURRENT STATUS</strong><br>";
            echo "• Using <strong>HARDCODED LIST</strong> ($hardcodedCount banks)<br>";
            echo "• API is not available or failing<br>";
            echo "<br>Recommendation: Keep using hardcoded list until API is properly configured.";
        }
        echo "</div>";
        ?>
        
        <hr>
        <p style="text-align: center; color: #666;">
            <small>Test performed on <?php echo date('Y-m-d H:i:s'); ?></small>
        </p>
    </div>
</body>
</html>

