<?php
/**
 * Test Script for All Fingpay AEPS Endpoints
 * 
 * Run this script to test all endpoints:
 * php test_all_endpoints.php
 */

// Colors for output (if terminal supports)
$GREEN = "\033[32m";
$RED = "\033[31m";
$YELLOW = "\033[33m";
$BLUE = "\033[34m";
$RESET = "\033[0m";

// Base URL for testing
$BASE_URL = 'http://localhost:8000';

echo "========================================\n";
echo "  Fingpay AEPS Endpoint Testing Tool\n";
echo "========================================\n\n";

// Helper function to make HTTP requests
function makeRequest($url, $method = 'GET', $data = null) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    
    if ($method === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($data !== null) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json'
            ]);
        }
    }
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    return [
        'response' => $response,
        'http_code' => $httpCode,
        'error' => $error
    ];
}

// Test 1: Bank List Endpoint
echo "${BLUE}[TEST 1]${RESET} Testing Bank List Endpoint...\n";
echo "URL: {$BASE_URL}/get_bank_list.php\n";
$result = makeRequest("{$BASE_URL}/get_bank_list.php");
if ($result['error']) {
    echo "${RED}✗ ERROR: ${result['error']}${RESET}\n";
} else {
    $data = json_decode($result['response'], true);
    if ($data && isset($data['status']) && $data['status'] === 'success') {
        echo "${GREEN}✓ SUCCESS${RESET}\n";
        echo "  Status: {$data['status']}\n";
        echo "  Banks Count: " . ($data['count'] ?? count($data['banks'] ?? [])) . "\n";
        echo "  Source: " . ($data['source'] ?? 'unknown') . "\n";
        if (isset($data['banks'][0])) {
            echo "  Sample Bank: {$data['banks'][0]['code']} - {$data['banks'][0]['name']}\n";
        }
    } else {
        echo "${RED}✗ FAILED${RESET}\n";
        echo "  Response: " . substr($result['response'], 0, 200) . "\n";
    }
}
echo "\n";

// Test 2: Balance Inquiry
echo "${BLUE}[TEST 2]${RESET} Testing Balance Inquiry Endpoint...\n";
echo "URL: {$BASE_URL}/api_handler.php\n";
$balanceData = [
    'type' => 'balance',
    'data' => [
        'aadhaarNumber' => '123456789012',
        'bankCode' => '607094',
        'deviceIMEI' => '8982848'
    ]
];
$result = makeRequest("{$BASE_URL}/api_handler.php", 'POST', $balanceData);
if ($result['error']) {
    echo "${RED}✗ ERROR: ${result['error']}${RESET}\n";
} else {
    $data = json_decode($result['response'], true);
    if ($data && isset($data['status'])) {
        echo "${GREEN}✓ Request Processed${RESET}\n";
        echo "  Status: {$data['status']}\n";
        echo "  Message: " . ($data['message'] ?? 'N/A') . "\n";
        if (isset($data['transactionRef'])) {
            echo "  Transaction Ref: {$data['transactionRef']}\n";
        }
    } else {
        echo "${RED}✗ FAILED${RESET}\n";
        echo "  Response: " . substr($result['response'], 0, 200) . "\n";
    }
}
echo "\n";

// Test 3: Cash Withdrawal
echo "${BLUE}[TEST 3]${RESET} Testing Cash Withdrawal Endpoint...\n";
$withdrawalData = [
    'type' => 'withdrawal',
    'data' => [
        'aadhaarNumber' => '123456789012',
        'amount' => 100.00,
        'customerName' => 'Test User',
        'bankCode' => '607094',
        'deviceIMEI' => '8982848'
    ]
];
$result = makeRequest("{$BASE_URL}/api_handler.php", 'POST', $withdrawalData);
if ($result['error']) {
    echo "${RED}✗ ERROR: ${result['error']}${RESET}\n";
} else {
    $data = json_decode($result['response'], true);
    if ($data && isset($data['status'])) {
        echo "${GREEN}✓ Request Processed${RESET}\n";
        echo "  Status: {$data['status']}\n";
        echo "  Message: " . ($data['message'] ?? 'N/A') . "\n";
    } else {
        echo "${RED}✗ FAILED${RESET}\n";
    }
}
echo "\n";

// Test 4: Mini Statement
echo "${BLUE}[TEST 4]${RESET} Testing Mini Statement Endpoint...\n";
$statementData = [
    'type' => 'statement',
    'data' => [
        'aadhaarNumber' => '123456789012',
        'bankCode' => '607094',
        'deviceIMEI' => '8982848'
    ]
];
$result = makeRequest("{$BASE_URL}/api_handler.php", 'POST', $statementData);
if ($result['error']) {
    echo "${RED}✗ ERROR: ${result['error']}${RESET}\n";
} else {
    $data = json_decode($result['response'], true);
    if ($data && isset($data['status'])) {
        echo "${GREEN}✓ Request Processed${RESET}\n";
        echo "  Status: {$data['status']}\n";
        echo "  Message: " . ($data['message'] ?? 'N/A') . "\n";
    } else {
        echo "${RED}✗ FAILED${RESET}\n";
    }
}
echo "\n";

// Test 5: Cash Deposit
echo "${BLUE}[TEST 5]${RESET} Testing Cash Deposit Endpoint...\n";
$depositData = [
    'type' => 'deposit',
    'data' => [
        'aadhaarNumber' => '123456789012',
        'amount' => 500.00,
        'customerName' => 'Test User',
        'bankCode' => '607094',
        'deviceIMEI' => '8982848'
    ]
];
$result = makeRequest("{$BASE_URL}/api_handler.php", 'POST', $depositData);
if ($result['error']) {
    echo "${RED}✗ ERROR: ${result['error']}${RESET}\n";
} else {
    $data = json_decode($result['response'], true);
    if ($data && isset($data['status'])) {
        echo "${GREEN}✓ Request Processed${RESET}\n";
        echo "  Status: {$data['status']}\n";
        echo "  Message: " . ($data['message'] ?? 'N/A') . "\n";
    } else {
        echo "${RED}✗ FAILED${RESET}\n";
    }
}
echo "\n";

// Test 6: Configuration Check
echo "${BLUE}[TEST 6]${RESET} Checking Configuration...\n";
if (file_exists(__DIR__ . '/api_config.php')) {
    $config = require __DIR__ . '/api_config.php';
    echo "${GREEN}✓ Config file exists${RESET}\n";
    if (isset($config['credentials'])) {
        echo "  Merchant ID: " . ($config['credentials']['merchant_id'] ?? 'N/A') . "\n";
        echo "  Username: " . ($config['credentials']['username'] ?? 'N/A') . "\n";
        echo "  Device IMEI: " . ($config['credentials']['default_device_imei'] ?? 'N/A') . "\n";
    }
} else {
    echo "${RED}✗ Config file not found${RESET}\n";
}
echo "\n";

// Test 7: Certificate File Check
echo "${BLUE}[TEST 7]${RESET} Checking Certificate File...\n";
$certPath = __DIR__ . '/../fingpay_public_production.txt';
if (file_exists($certPath)) {
    echo "${GREEN}✓ Certificate file exists${RESET}\n";
    echo "  Path: {$certPath}\n";
    echo "  Size: " . filesize($certPath) . " bytes\n";
} else {
    echo "${YELLOW}⚠ Certificate file not found${RESET}\n";
    echo "  Path: {$certPath}\n";
    echo "  Update path in api_config.php if different\n";
}
echo "\n";

echo "========================================\n";
echo "  Testing Complete!\n";
echo "========================================\n";
echo "\n";
echo "Next Steps:\n";
echo "1. Start PHP server: php -S localhost:8000\n";
echo "2. Open browser: http://localhost:8000/index.html\n";
echo "3. Test each endpoint in the web interface\n";
echo "\n";

