<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Fingpay AEPS - Transaction Portal</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
        }

        .header {
            text-align: center;
            color: white;
            margin-bottom: 30px;
        }

        .header h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
        }

        .tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            flex-wrap: wrap;
        }

        .tab-button {
            padding: 12px 24px;
            background: rgba(255, 255, 255, 0.2);
            color: white;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            font-size: 16px;
            transition: all 0.3s;
            backdrop-filter: blur(10px);
        }

        .tab-button:hover {
            background: rgba(255, 255, 255, 0.3);
        }

        .tab-button.active {
            background: white;
            color: #667eea;
            font-weight: bold;
        }

        .tab-content {
            display: none;
            background: white;
            padding: 30px;
            border-radius: 12px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
        }

        .tab-content.active {
            display: block;
            animation: fadeIn 0.3s;
        }

        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .form-group {
            margin-bottom: 20px;
        }

        label {
            display: block;
            margin-bottom: 8px;
            color: #333;
            font-weight: 600;
            font-size: 14px;
        }

        input[type="text"],
        input[type="number"],
        input[type="tel"],
        select {
            width: 100%;
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.3s;
        }

        input:focus,
        select:focus {
            outline: none;
            border-color: #667eea;
        }

        .required {
            color: red;
        }

        .btn-submit {
            width: 100%;
            padding: 14px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 18px;
            font-weight: bold;
            cursor: pointer;
            transition: transform 0.2s;
            margin-top: 10px;
        }

        .btn-submit:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }

        .btn-submit:active {
            transform: translateY(0);
        }

        .response-container {
            margin-top: 20px;
            padding: 20px;
            border-radius: 8px;
            display: none;
        }

        .response-container.show {
            display: block;
            animation: slideDown 0.3s;
        }

        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .response-success {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
        }

        .response-error {
            background: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
        }

        .response-content {
            white-space: pre-wrap;
            font-family: 'Courier New', monospace;
            font-size: 14px;
            line-height: 1.6;
        }

        .loading {
            display: none;
            text-align: center;
            padding: 20px;
        }

        .loading.show {
            display: block;
        }

        .spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #667eea;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 0 auto;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        /* Bank List Styles */
        .bank-search-container {
            margin-bottom: 25px;
        }

        .bank-search-input {
            width: 100%;
            padding: 14px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.3s;
        }

        .bank-search-input:focus {
            outline: none;
            border-color: #667eea;
        }

        .bank-list-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            flex-wrap: wrap;
            gap: 10px;
        }

        .bank-count {
            color: #666;
            font-size: 14px;
        }

        .bank-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 15px;
            margin-top: 20px;
        }

        .bank-card {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            padding: 20px;
            border-radius: 10px;
            border: 2px solid #e0e0e0;
            transition: all 0.3s;
            cursor: pointer;
        }

        .bank-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 20px rgba(102, 126, 234, 0.3);
            border-color: #667eea;
        }

        .bank-code {
            font-size: 24px;
            font-weight: bold;
            color: #667eea;
            margin-bottom: 8px;
            font-family: 'Courier New', monospace;
        }

        .bank-name {
            font-size: 16px;
            color: #333;
            line-height: 1.4;
        }

        .bank-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }

        .bank-table thead {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }

        .bank-table th,
        .bank-table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #e0e0e0;
        }

        .bank-table th {
            font-weight: 600;
            font-size: 14px;
        }

        .bank-table tbody tr:hover {
            background-color: #f5f7fa;
        }

        .bank-table tbody tr:last-child td {
            border-bottom: none;
        }

        .bank-code-cell {
            font-family: 'Courier New', monospace;
            font-weight: bold;
            color: #667eea;
            font-size: 16px;
        }

        .view-toggle {
            display: flex;
            gap: 10px;
            margin-bottom: 15px;
        }

        .view-btn {
            padding: 8px 16px;
            background: rgba(102, 126, 234, 0.1);
            border: 2px solid #667eea;
            border-radius: 6px;
            color: #667eea;
            cursor: pointer;
            font-size: 14px;
            transition: all 0.3s;
        }

        .view-btn:hover {
            background: rgba(102, 126, 234, 0.2);
        }

        .view-btn.active {
            background: #667eea;
            color: white;
        }

        .no-banks {
            text-align: center;
            padding: 40px;
            color: #999;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🏦 Fingpay AEPS Portal</h1>
            <p>Secure Aadhaar Enabled Payment System</p>
        </div>

        <div class="tabs">
            <button class="tab-button active" onclick="showTab('balance')">Balance Inquiry</button>
            <button class="tab-button" onclick="showTab('withdrawal')">Cash Withdrawal</button>
            <button class="tab-button" onclick="showTab('statement')">Mini Statement</button>
            <button class="tab-button" onclick="showTab('deposit')">Cash Deposit</button>
            <button class="tab-button" onclick="showTab('banks')">Aadhaar Banks</button>
        </div>

        <div style="text-align: center; margin-bottom: 20px;">
            <!-- User Onboarding removed - Not in documentation -->
            <a href="merchant_onboard.php" style="display: inline-block; margin: 5px; padding: 10px 20px; background: rgba(255, 255, 255, 0.2); color: white; text-decoration: none; border-radius: 8px; transition: background 0.3s;">🏪 Merchant Onboarding</a>
            <a href="ekyc_verify.php" style="display: inline-block; margin: 5px; padding: 10px 20px; background: rgba(255, 255, 255, 0.2); color: white; text-decoration: none; border-radius: 8px; transition: background 0.3s;">🆔 EKYC Verification</a>
        </div>

        <!-- Balance Inquiry Tab -->
        <div id="balance" class="tab-content active">
            <h2>Balance Inquiry</h2>
            <form id="balanceForm">
                <div class="form-group">
                    <label>Aadhaar Number <span class="required">*</span></label>
                    <input type="tel" name="aadhaarNumber" maxlength="12" pattern="[0-9]{12}" required placeholder="Enter 12 digit Aadhaar number">
                </div>
                <div class="form-group">
                    <label>Bank <span class="required">*</span></label>
                    <select name="bankCode" id="balanceBankCode" required>
                        <option value="">Select Bank</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Device IMEI (Optional)</label>
                    <input type="text" name="deviceIMEI" placeholder="Enter device IMEI">
                </div>
                <button type="submit" class="btn-submit">Check Balance</button>
            </form>
            <div class="loading" id="balanceLoading">
                <div class="spinner"></div>
                <p>Fetching balance...</p>
            </div>
            <div class="response-container" id="balanceResponse"></div>
        </div>

        <!-- Cash Withdrawal Tab -->
        <div id="withdrawal" class="tab-content">
            <h2>Cash Withdrawal</h2>
            <form id="withdrawalForm">
                <div class="form-group">
                    <label>Aadhaar Number <span class="required">*</span></label>
                    <input type="tel" name="aadhaarNumber" maxlength="12" pattern="[0-9]{12}" required placeholder="Enter 12 digit Aadhaar number">
                </div>
                <div class="form-group">
                    <label>Amount <span class="required">*</span></label>
                    <input type="number" name="amount" step="0.01" min="0" required placeholder="Enter amount">
                </div>
                <div class="form-group">
                    <label>Customer Name <span class="required">*</span></label>
                    <input type="text" name="customerName" required placeholder="Enter customer name">
                </div>
                <div class="form-group">
                    <label>Bank <span class="required">*</span></label>
                    <select name="bankCode" id="withdrawalBankCode" required>
                        <option value="">Select Bank</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Device IMEI (Optional)</label>
                    <input type="text" name="deviceIMEI" placeholder="Enter device IMEI">
                </div>
                <button type="submit" class="btn-submit">Process Withdrawal</button>
            </form>
            <div class="loading" id="withdrawalLoading">
                <div class="spinner"></div>
                <p>Processing transaction...</p>
            </div>
            <div class="response-container" id="withdrawalResponse"></div>
        </div>

        <!-- Mini Statement Tab -->
        <div id="statement" class="tab-content">
            <h2>Mini Statement</h2>
            <form id="statementForm">
                <div class="form-group">
                    <label>Aadhaar Number <span class="required">*</span></label>
                    <input type="tel" name="aadhaarNumber" maxlength="12" pattern="[0-9]{12}" required placeholder="Enter 12 digit Aadhaar number">
                </div>
                <div class="form-group">
                    <label>Bank <span class="required">*</span></label>
                    <select name="bankCode" id="statementBankCode" required>
                        <option value="">Select Bank</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Device IMEI (Optional)</label>
                    <input type="text" name="deviceIMEI" placeholder="Enter device IMEI">
                </div>
                <button type="submit" class="btn-submit">Get Statement</button>
            </form>
            <div class="loading" id="statementLoading">
                <div class="spinner"></div>
                <p>Fetching statement...</p>
            </div>
            <div class="response-container" id="statementResponse"></div>
        </div>

        <!-- Cash Deposit Tab -->
        <div id="deposit" class="tab-content">
            <h2>Cash Deposit</h2>
            <form id="depositForm">
                <div class="form-group">
                    <label>Aadhaar Number <span class="required">*</span></label>
                    <input type="tel" name="aadhaarNumber" maxlength="12" pattern="[0-9]{12}" required placeholder="Enter 12 digit Aadhaar number">
                </div>
                <div class="form-group">
                    <label>Amount <span class="required">*</span></label>
                    <input type="number" name="amount" step="0.01" min="0" required placeholder="Enter amount">
                </div>
                <div class="form-group">
                    <label>Customer Name <span class="required">*</span></label>
                    <input type="text" name="customerName" required placeholder="Enter customer name">
                </div>
                <div class="form-group">
                    <label>Bank <span class="required">*</span></label>
                    <select name="bankCode" id="depositBankCode" required>
                        <option value="">Select Bank</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Device IMEI (Optional)</label>
                    <input type="text" name="deviceIMEI" placeholder="Enter device IMEI">
                </div>
                <button type="submit" class="btn-submit">Process Deposit</button>
            </form>
            <div class="loading" id="depositLoading">
                <div class="spinner"></div>
                <p>Processing deposit...</p>
            </div>
            <div class="response-container" id="depositResponse"></div>
        </div>

        <!-- Aadhaar Banks Tab -->
        <div id="banks" class="tab-content">
            <h2>🏦 Aadhaar Enabled Banks</h2>
            <p style="color: #666; margin-bottom: 20px;">List of all banks supported for Aadhaar Enabled Payment System (AEPS) transactions</p>
            
            <div class="bank-search-container">
                <input type="text" 
                       id="bankSearch" 
                       class="bank-search-input" 
                       placeholder="🔍 Search banks by name or code...">
            </div>

            <div class="bank-list-header">
                <div class="bank-count">
                    <span id="bankCount">0</span> banks available
                </div>
                <div class="view-toggle">
                    <button class="view-btn active" onclick="setViewMode('grid')">Grid View</button>
                    <button class="view-btn" onclick="setViewMode('table')">Table View</button>
                </div>
            </div>

            <div id="bankGridContainer" class="bank-grid"></div>
            <div id="bankTableContainer" style="display: none;">
                <table class="bank-table">
                    <thead>
                        <tr>
                            <th>Bank Code</th>
                            <th>Bank Name</th>
                        </tr>
                    </thead>
                    <tbody id="bankTableBody"></tbody>
                </table>
            </div>
            <div id="noBanksMessage" class="no-banks" style="display: none;">
                No banks found matching your search criteria.
            </div>
        </div>
    </div>

    <script>
        // Load bank list on page load
        let bankList = [];
        
        let currentViewMode = 'grid';
        let filteredBanks = [];

        fetch('get_bank_list.php')
            .then(response => response.json())
            .then(data => {
                if (data.status === 'success' && data.banks) {
                    bankList = data.banks;
                    filteredBanks = bankList;
                    populateBankDropdowns();
                    displayBanks();
                }
            })
            .catch(error => {
                console.error('Error loading bank list:', error);
            });
        
        function populateBankDropdowns() {
            const dropdowns = [
                'withdrawalBankCode',
                'balanceBankCode',
                'statementBankCode',
                'depositBankCode'
            ];
            
            dropdowns.forEach(dropdownId => {
                const dropdown = document.getElementById(dropdownId);
                if (dropdown) {
                    // Clear existing options except the first one
                    dropdown.innerHTML = '<option value="">Select Bank</option>';
                    
                    // Add bank options
                    bankList.forEach(bank => {
                        const option = document.createElement('option');
                        option.value = bank.code;
                        option.textContent = bank.name;
                        dropdown.appendChild(option);
                    });
                }
            });
        }
        
        function showTab(tabName) {
            // Hide all tabs
            document.querySelectorAll('.tab-content').forEach(tab => {
                tab.classList.remove('active');
            });
            document.querySelectorAll('.tab-button').forEach(btn => {
                btn.classList.remove('active');
            });

            // Show selected tab
            document.getElementById(tabName).classList.add('active');
            event.target.classList.add('active');
        }

        // Cash Withdrawal Form Handler
        document.getElementById('withdrawalForm').addEventListener('submit', function(e) {
            e.preventDefault();
            const formData = new FormData(this);
            const data = Object.fromEntries(formData);
            submitTransaction('withdrawal', data);
        });

        // Balance Inquiry Form Handler
        document.getElementById('balanceForm').addEventListener('submit', function(e) {
            e.preventDefault();
            const formData = new FormData(this);
            const data = Object.fromEntries(formData);
            submitTransaction('balance', data);
        });

        // Mini Statement Form Handler
        document.getElementById('statementForm').addEventListener('submit', function(e) {
            e.preventDefault();
            const formData = new FormData(this);
            const data = Object.fromEntries(formData);
            submitTransaction('statement', data);
        });

        // Cash Deposit Form Handler
        document.getElementById('depositForm').addEventListener('submit', function(e) {
            e.preventDefault();
            const formData = new FormData(this);
            const data = Object.fromEntries(formData);
            submitTransaction('deposit', data);
        });

        function submitTransaction(type, data) {
            const loadingId = type + 'Loading';
            const responseId = type + 'Response';

            // Show loading
            document.getElementById(loadingId).classList.add('show');
            document.getElementById(responseId).classList.remove('show');

            // Send AJAX request
            fetch('api_handler.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    type: type,
                    data: data
                })
            })
            .then(response => response.json())
            .then(result => {
                document.getElementById(loadingId).classList.remove('show');
                const responseDiv = document.getElementById(responseId);
                responseDiv.classList.add('show');
                
                if (result.status === 'success') {
                    responseDiv.className = 'response-container show response-success';
                    responseDiv.innerHTML = '<div class="response-content">' + JSON.stringify(result.data, null, 2) + '</div>';
                } else {
                    responseDiv.className = 'response-container show response-error';
                    responseDiv.innerHTML = '<div class="response-content">' + (result.message || JSON.stringify(result, null, 2)) + '</div>';
                }
            })
            .catch(error => {
                document.getElementById(loadingId).classList.remove('show');
                const responseDiv = document.getElementById(responseId);
                responseDiv.classList.add('show');
                responseDiv.className = 'response-container show response-error';
                responseDiv.innerHTML = '<div class="response-content">Error: ' + error.message + '</div>';
            });
        }

        // Bank List Functions
        function displayBanks() {
            updateBankCount();
            
            if (filteredBanks.length === 0) {
                document.getElementById('bankGridContainer').style.display = 'none';
                document.getElementById('bankTableContainer').style.display = 'none';
                document.getElementById('noBanksMessage').style.display = 'block';
                return;
            }

            document.getElementById('noBanksMessage').style.display = 'none';

            if (currentViewMode === 'grid') {
                displayBankGrid();
            } else {
                displayBankTable();
            }
        }

        function displayBankGrid() {
            const gridContainer = document.getElementById('bankGridContainer');
            gridContainer.innerHTML = '';
            
            filteredBanks.forEach(bank => {
                const card = document.createElement('div');
                card.className = 'bank-card';
                card.innerHTML = `
                    <div class="bank-code">${bank.code}</div>
                    <div class="bank-name">${bank.name}</div>
                `;
                card.onclick = () => copyBankCode(bank.code, bank.name);
                gridContainer.appendChild(card);
            });

            document.getElementById('bankGridContainer').style.display = 'grid';
            document.getElementById('bankTableContainer').style.display = 'none';
        }

        function displayBankTable() {
            const tableBody = document.getElementById('bankTableBody');
            tableBody.innerHTML = '';
            
            filteredBanks.forEach(bank => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td class="bank-code-cell">${bank.code}</td>
                    <td>${bank.name}</td>
                `;
                row.onclick = () => copyBankCode(bank.code, bank.name);
                row.style.cursor = 'pointer';
                tableBody.appendChild(row);
            });

            document.getElementById('bankGridContainer').style.display = 'none';
            document.getElementById('bankTableContainer').style.display = 'block';
        }

        function updateBankCount() {
            document.getElementById('bankCount').textContent = filteredBanks.length;
        }

        function setViewMode(mode) {
            currentViewMode = mode;
            
            // Update button states
            document.querySelectorAll('.view-btn').forEach(btn => {
                btn.classList.remove('active');
                if ((mode === 'grid' && btn.textContent.includes('Grid')) || 
                    (mode === 'table' && btn.textContent.includes('Table'))) {
                    btn.classList.add('active');
                }
            });
            
            displayBanks();
        }

        function filterBanks(searchTerm) {
            if (!searchTerm.trim()) {
                filteredBanks = bankList;
            } else {
                const term = searchTerm.toLowerCase();
                filteredBanks = bankList.filter(bank => 
                    bank.name.toLowerCase().includes(term) || 
                    bank.code.includes(term)
                );
            }
            displayBanks();
        }

        function copyBankCode(code, name) {
            // Copy bank code to clipboard
            navigator.clipboard.writeText(code).then(() => {
                // Show temporary notification
                const notification = document.createElement('div');
                notification.style.cssText = `
                    position: fixed;
                    top: 20px;
                    right: 20px;
                    background: #667eea;
                    color: white;
                    padding: 15px 25px;
                    border-radius: 8px;
                    box-shadow: 0 4px 12px rgba(0,0,0,0.2);
                    z-index: 1000;
                    animation: slideIn 0.3s;
                `;
                notification.textContent = `Bank code ${code} copied! (${name})`;
                document.body.appendChild(notification);
                
                setTimeout(() => {
                    notification.style.animation = 'slideOut 0.3s';
                    setTimeout(() => notification.remove(), 300);
                }, 2000);
            });
        }

        // Search functionality
        document.addEventListener('DOMContentLoaded', function() {
            const searchInput = document.getElementById('bankSearch');
            if (searchInput) {
                searchInput.addEventListener('input', function(e) {
                    filterBanks(e.target.value);
                });
            }
        });

        // Add CSS animation for notification
        const style = document.createElement('style');
        style.textContent = `
            @keyframes slideIn {
                from {
                    transform: translateX(100%);
                    opacity: 0;
                }
                to {
                    transform: translateX(0);
                    opacity: 1;
                }
            }
            @keyframes slideOut {
                from {
                    transform: translateX(0);
                    opacity: 1;
                }
                to {
                    transform: translateX(100%);
                    opacity: 0;
                }
            }
        `;
        document.head.appendChild(style);
    </script>
</body>
</html>

