<?php
/**
 * Actual Bank List Comparison
 * This script compares API bank list with hardcoded bank list
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once 'bank_list.php';

// Load API configuration
$configFile = __DIR__ . '/api_config.php';
if (file_exists($configFile)) {
    $config = require $configFile;
    $credentials = $config['credentials'] ?? [];
    $defaultMerchantId = $credentials['merchant_id'] ?? $credentials['super_merchant_id'] ?? null;
    $certPath = $config['certificate_path'] ?? (__DIR__ . '/../fingpay_public_production.txt');
} else {
    $defaultMerchantId = null;
    $certPath = __DIR__ . '/../fingpay_public_production.txt';
}

echo "=== BANK LIST COMPARISON REPORT ===\n\n";

// 1. Get Hardcoded List
echo "1. HARDCODED LIST:\n";
$hardcodedList = getBankList($certPath, $defaultMerchantId, false);
echo "   Total Banks: " . count($hardcodedList) . "\n";
echo "   Banks: ";
foreach ($hardcodedList as $code => $name) {
    echo "$code, ";
}
echo "\n\n";

// 2. Try to get API List
echo "2. API LIST (Testing...):\n";
$apiList = [];
$apiSuccess = false;
$apiError = '';

try {
    $apiList = getBankList($certPath, $defaultMerchantId, true);
    if (!empty($apiList)) {
        $apiSuccess = true;
        echo "   ✅ API Success!\n";
        echo "   Total Banks from API: " . count($apiList) . "\n";
        echo "   Banks: ";
        foreach ($apiList as $code => $name) {
            echo "$code, ";
        }
        echo "\n\n";
    } else {
        $apiError = "API returned empty list";
        echo "   ❌ API Failed: $apiError\n\n";
    }
} catch (Exception $e) {
    $apiError = $e->getMessage();
    echo "   ❌ API Error: " . $apiError . "\n\n";
}

// 3. Comparison
echo "3. COMPARISON RESULTS:\n";
echo str_repeat("-", 50) . "\n";

if ($apiSuccess && !empty($apiList)) {
    // Compare both lists
    $allCodes = array_unique(array_merge(array_keys($hardcodedList), array_keys($apiList)));
    $matches = 0;
    $exactMatches = 0;
    $differences = [];
    $onlyHardcoded = [];
    $onlyAPI = [];
    
    foreach ($allCodes as $code) {
        $hardcodedName = $hardcodedList[$code] ?? null;
        $apiName = $apiList[$code] ?? null;
        
        if ($hardcodedName && $apiName) {
            $matches++;
            if ($hardcodedName === $apiName) {
                $exactMatches++;
            } else {
                $differences[] = [
                    'code' => $code,
                    'hardcoded' => $hardcodedName,
                    'api' => $apiName
                ];
            }
        } elseif ($hardcodedName && !$apiName) {
            $onlyHardcoded[] = ['code' => $code, 'name' => $hardcodedName];
        } elseif (!$hardcodedName && $apiName) {
            $onlyAPI[] = ['code' => $code, 'name' => $apiName];
        }
    }
    
    echo "   Hardcoded Banks: " . count($hardcodedList) . "\n";
    echo "   API Banks: " . count($apiList) . "\n";
    echo "   ✅ Exact Matches (same code & name): $exactMatches\n";
    echo "   ⚠️  Same Code, Different Names: " . count($differences) . "\n";
    echo "   ❌ Only in Hardcoded: " . count($onlyHardcoded) . "\n";
    echo "   ➕ Only in API: " . count($onlyAPI) . "\n\n";
    
    // Check if lists are identical
    $areSame = (count($differences) == 0 && count($onlyHardcoded) == 0 && count($onlyAPI) == 0 && $exactMatches == count($hardcodedList) && $exactMatches == count($apiList));
    
    if ($areSame) {
        echo "   🎉 RESULT: PERFECT MATCH!\n";
        echo "   ✅ Hardcoded और API lists बिल्कुल SAME हैं!\n";
        echo "   ✅ All banks match perfectly!\n\n";
    } else {
        echo "   ⚠️  RESULT: LISTS ARE DIFFERENT\n\n";
        
        if (!empty($differences)) {
            echo "   Banks with Different Names:\n";
            foreach ($differences as $diff) {
                echo "     - Code: {$diff['code']}\n";
                echo "       Hardcoded: {$diff['hardcoded']}\n";
                echo "       API: {$diff['api']}\n";
            }
            echo "\n";
        }
        
        if (!empty($onlyHardcoded)) {
            echo "   Banks Only in Hardcoded (" . count($onlyHardcoded) . "):\n";
            foreach (array_slice($onlyHardcoded, 0, 5) as $bank) {
                echo "     - {$bank['code']}: {$bank['name']}\n";
            }
            if (count($onlyHardcoded) > 5) {
                echo "     ... and " . (count($onlyHardcoded) - 5) . " more\n";
            }
            echo "\n";
        }
        
        if (!empty($onlyAPI)) {
            echo "   Banks Only in API (" . count($onlyAPI) . "):\n";
            foreach (array_slice($onlyAPI, 0, 5) as $bank) {
                echo "     - {$bank['code']}: {$bank['name']}\n";
            }
            if (count($onlyAPI) > 5) {
                echo "     ... and " . (count($onlyAPI) - 5) . " more\n";
            }
            echo "\n";
        }
    }
    
} else {
    echo "   ❌ Cannot compare: API is not available\n";
    echo "   📋 Currently using HARDCODED LIST with " . count($hardcodedList) . " banks\n\n";
}

// 4. Final Summary
echo "4. FINAL SUMMARY:\n";
echo str_repeat("-", 50) . "\n";
echo "   Current Setting: \$useAPI = false (HARDCODED)\n";
echo "   Hardcoded Banks Available: " . count($hardcodedList) . "\n";

if ($apiSuccess) {
    echo "   API Status: ✅ Working\n";
    echo "   API Banks Available: " . count($apiList) . "\n";
    if (isset($areSame) && $areSame) {
        echo "   ✅ RECOMMENDATION: Both lists are identical. Perfect!\n";
    } else {
        echo "   ⚠️  RECOMMENDATION: Lists differ. Review needed.\n";
    }
} else {
    echo "   API Status: ❌ Not available / Failed\n";
    echo "   ✅ RECOMMENDATION: Keep using hardcoded list.\n";
}

echo "\n" . str_repeat("=", 50) . "\n";

