# Testing Guide - Fingpay AEPS Application

This guide explains how to run and test all endpoints of the Fingpay AEPS application.

## Prerequisites

### 1. PHP Installation
- **PHP Version**: 7.0 or higher (7.4+ recommended)
- **Required Extensions**:
  - OpenSSL extension
  - cURL extension
  - JSON extension

### 2. Web Server
Choose one of the following:
- **Built-in PHP Server** (easiest for testing)
- **Apache** with PHP module
- **Nginx** with PHP-FPM
- **XAMPP/WAMP** (Windows)

### 3. Certificate File
Ensure the Fingpay public certificate file exists at:
```
/fingpay_public_production.txt
```
or update the path in `api_config.php` if using a different location.

## Setup Instructions

### Step 1: Verify File Structure
```
AEPS/
├── code/
│   ├── api_config.php          ✅ Credentials configured
│   ├── api_handler.php          ✅ API handler
│   ├── FingpayAEPS.php         ✅ Core API class
│   ├── bank_list.php            ✅ Bank list functions
│   ├── get_bank_list.php        ✅ Bank list endpoint
│   ├── index.html               ✅ Web interface
│   └── ...
├── fingpay_public_production.txt ✅ Certificate file
└── ...
```

### Step 2: Check Configuration
Open `code/api_config.php` and verify:
- ✅ Credentials are set correctly
- ✅ Certificate path is correct
- ✅ API endpoints are configured

### Step 3: Start the Server

#### Option A: Built-in PHP Server (Recommended for Testing)
```bash
# Navigate to the code directory
cd code

# Start PHP built-in server
php -S localhost:8000

# Server will start at: http://localhost:8000
```

#### Option B: XAMPP/WAMP
1. Copy the `code` folder to `htdocs` (XAMPP) or `www` (WAMP)
2. Start Apache server
3. Access via: `http://localhost/code/index.html`

#### Option C: Apache/Nginx
Configure virtual host to point to the `code` directory

## Testing Methods

### Method 1: Web Interface (Easiest)

1. **Start the server** (if using built-in PHP server):
   ```bash
   cd code
   php -S localhost:8000
   ```

2. **Open browser** and navigate to:
   ```
   http://localhost:8000/index.html
   ```

3. **Test each tab**:
   - **Cash Withdrawal**: Fill form and submit
   - **Balance Inquiry**: Enter Aadhaar and bank code
   - **Mini Statement**: Get transaction history
   - **Cash Deposit**: Deposit cash
   - **Aadhaar Banks**: View all supported banks

4. **Check Browser Console** (F12):
   - Look for JavaScript errors
   - Check network requests
   - Verify API responses

### Method 2: Direct API Testing (Using cURL)

#### Test Bank List Endpoint
```bash
# Test bank list endpoint
curl -X GET "http://localhost:8000/get_bank_list.php"

# Or with merchant ID
curl -X GET "http://localhost:8000/get_bank_list.php?merchantId=1143"
```

#### Test Cash Withdrawal
```bash
curl -X POST "http://localhost:8000/api_handler.php" \
  -H "Content-Type: application/json" \
  -d '{
    "type": "withdrawal",
    "data": {
      "aadhaarNumber": "123456789012",
      "amount": 100.00,
      "customerName": "Test User",
      "bankCode": "607094",
      "deviceIMEI": "8982848"
    }
  }'
```

#### Test Balance Inquiry
```bash
curl -X POST "http://localhost:8000/api_handler.php" \
  -H "Content-Type: application/json" \
  -d '{
    "type": "balance",
    "data": {
      "aadhaarNumber": "123456789012",
      "bankCode": "607094",
      "deviceIMEI": "8982848"
    }
  }'
```

#### Test Mini Statement
```bash
curl -X POST "http://localhost:8000/api_handler.php" \
  -H "Content-Type: application/json" \
  -d '{
    "type": "statement",
    "data": {
      "aadhaarNumber": "123456789012",
      "bankCode": "607094",
      "deviceIMEI": "8982848"
    }
  }'
```

#### Test Cash Deposit
```bash
curl -X POST "http://localhost:8000/api_handler.php" \
  -H "Content-Type: application/json" \
  -d '{
    "type": "deposit",
    "data": {
      "aadhaarNumber": "123456789012",
      "amount": 500.00,
      "customerName": "Test User",
      "bankCode": "607094",
      "deviceIMEI": "8982848"
    }
  }'
```

### Method 3: PHP Command Line Testing

Create a test file `test_endpoints.php`:

```php
<?php
require_once 'api_handler.php';
// Test each endpoint programmatically
```

## Step-by-Step Testing Checklist

### ✅ 1. Server Setup
- [ ] PHP server running
- [ ] Port 8000 accessible (or configured port)
- [ ] Certificate file exists

### ✅ 2. Configuration Check
- [ ] `api_config.php` has correct credentials
- [ ] Certificate path is valid
- [ ] Merchant ID is 1143

### ✅ 3. Bank List Endpoint
- [ ] Open: `http://localhost:8000/get_bank_list.php`
- [ ] Should return JSON with bank list
- [ ] Check response format:
  ```json
  {
    "status": "success",
    "banks": [...],
    "source": "api" or "static",
    "count": 35
  }
  ```

### ✅ 4. Web Interface
- [ ] Open: `http://localhost:8000/index.html`
- [ ] All tabs visible
- [ ] Bank dropdowns populated
- [ ] Aadhaar Banks tab shows banks

### ✅ 5. Balance Inquiry
- [ ] Enter valid 12-digit Aadhaar number
- [ ] Select bank from dropdown
- [ ] Click "Check Balance"
- [ ] Check response (may show API error if using test Aadhaar)

### ✅ 6. Cash Withdrawal
- [ ] Fill all required fields
- [ ] Enter amount
- [ ] Submit form
- [ ] Check response

### ✅ 7. Mini Statement
- [ ] Enter Aadhaar and bank
- [ ] Submit
- [ ] Check response

### ✅ 8. Cash Deposit
- [ ] Fill required fields
- [ ] Submit
- [ ] Check response

## Testing Each Endpoint

### Endpoint 1: Bank List (`get_bank_list.php`)

**URL**: `http://localhost:8000/get_bank_list.php`

**Method**: GET

**Expected Response**:
```json
{
  "status": "success",
  "banks": [
    {
      "code": "607094",
      "name": "State Bank of India (SBI)"
    },
    ...
  ],
  "source": "api",
  "count": 35
}
```

**Test in Browser**:
```
http://localhost:8000/get_bank_list.php
```

**Test with cURL**:
```bash
curl http://localhost:8000/get_bank_list.php
```

### Endpoint 2: Cash Withdrawal (`api_handler.php`)

**URL**: `http://localhost:8000/api_handler.php`

**Method**: POST

**Request Format**:
```json
{
  "type": "withdrawal",
  "data": {
    "aadhaarNumber": "123456789012",
    "amount": 100.00,
    "customerName": "John Doe",
    "bankCode": "607094",
    "deviceIMEI": "8982848"
  }
}
```

**Expected Response**:
```json
{
  "status": "success" or "error",
  "message": "...",
  "data": {...},
  "transactionRef": "TXN..."
}
```

### Endpoint 3: Balance Inquiry (`api_handler.php`)

**URL**: `http://localhost:8000/api_handler.php`

**Method**: POST

**Request Format**:
```json
{
  "type": "balance",
  "data": {
    "aadhaarNumber": "123456789012",
    "bankCode": "607094",
    "deviceIMEI": "8982848"
  }
}
```

### Endpoint 4: Mini Statement (`api_handler.php`)

**URL**: `http://localhost:8000/api_handler.php`

**Method**: POST

**Request Format**:
```json
{
  "type": "statement",
  "data": {
    "aadhaarNumber": "123456789012",
    "bankCode": "607094",
    "deviceIMEI": "8982848"
  }
}
```

### Endpoint 5: Cash Deposit (`api_handler.php`)

**URL**: `http://localhost:8000/api_handler.php`

**Method**: POST

**Request Format**:
```json
{
  "type": "deposit",
  "data": {
    "aadhaarNumber": "123456789012",
    "amount": 500.00,
    "customerName": "John Doe",
    "bankCode": "607094",
    "deviceIMEI": "8982848"
  }
}
```

## Troubleshooting

### Issue: Server Not Starting

**Error**: `php: command not found`

**Solution**:
```bash
# Windows (if PHP is installed)
# Add PHP to PATH environment variable

# Or use full path:
C:\php\php.exe -S localhost:8000

# Check PHP version:
php -v
```

### Issue: Certificate File Not Found

**Error**: `Public certificate file not found`

**Solution**:
1. Check if file exists: `fingpay_public_production.txt`
2. Update path in `api_config.php`:
   ```php
   'certificate_path' => __DIR__ . '/../fingpay_public_production.txt',
   ```
3. Verify file permissions (readable)

### Issue: CORS Errors

**Error**: `Access-Control-Allow-Origin` errors

**Solution**:
- Headers are already set in `api_handler.php`
- If issues persist, ensure server is running properly
- Check browser console for specific error

### Issue: API Returns Error

**Common Errors**:
1. **Invalid Aadhaar**: Use valid 12-digit Aadhaar number
2. **Bank Code Invalid**: Select from dropdown
3. **Network Error**: Check internet connection
4. **Authentication Error**: Verify credentials in `api_config.php`
5. **Certificate Error**: Verify certificate file and path

### Issue: Bank List Not Loading

**Check**:
1. Open browser console (F12)
2. Check network tab for `get_bank_list.php` request
3. Verify response format
4. Check if API is accessible or using fallback

### Issue: Transactions Failing

**Check**:
1. Verify credentials in `api_config.php`
2. Check certificate file exists
3. Verify Aadhaar number format (12 digits)
4. Check bank code is valid
5. Review API response for specific error message

## Testing with Real Data

⚠️ **Important**: When testing with real data:
- Use actual Aadhaar numbers (test numbers may not work)
- Verify bank codes match account banks
- Ensure account is linked to Aadhaar
- Test in sandbox/test environment first if available

## Quick Test Script

Create `test_all.php` in the `code` directory:

```php
<?php
/**
 * Quick test script for all endpoints
 */

echo "=== Testing Fingpay AEPS Endpoints ===\n\n";

// Test 1: Bank List
echo "1. Testing Bank List...\n";
$banks = file_get_contents('http://localhost:8000/get_bank_list.php');
echo "Response: " . substr($banks, 0, 100) . "...\n\n";

// Test 2: Balance Inquiry
echo "2. Testing Balance Inquiry...\n";
$ch = curl_init('http://localhost:8000/api_handler.php');
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
    'type' => 'balance',
    'data' => [
        'aadhaarNumber' => '123456789012',
        'bankCode' => '607094',
        'deviceIMEI' => '8982848'
    ]
]));
curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
$response = curl_exec($ch);
echo "Response: " . substr($response, 0, 200) . "...\n\n";

echo "Testing complete!\n";
```

Run:
```bash
php test_all.php
```

## Monitoring and Logs

### Check PHP Error Logs
- **Linux/Mac**: Check `/var/log/php_errors.log` or `php.ini` location
- **Windows**: Check error log path in `php.ini`
- **XAMPP**: `C:\xampp\php\logs\php_error_log`

### Enable Error Display (Development Only)
Add to top of PHP files for debugging:
```php
error_reporting(E_ALL);
ini_set('display_errors', 1);
```

## Next Steps After Testing

1. ✅ Verify all endpoints respond correctly
2. ✅ Test with real Aadhaar numbers (if available)
3. ✅ Verify transaction responses
4. ✅ Check error handling
5. ✅ Test with different banks
6. ✅ Verify bank list updates from API

## Production Deployment

Before deploying to production:
- [ ] Remove debug code
- [ ] Set proper error reporting
- [ ] Use HTTPS
- [ ] Secure `api_config.php` (restrict access)
- [ ] Set proper file permissions
- [ ] Enable logging
- [ ] Test all endpoints thoroughly

## Support

If you encounter issues:
1. Check browser console for JavaScript errors
2. Check PHP error logs
3. Verify configuration in `api_config.php`
4. Test endpoints individually using cURL
5. Review API responses for error details

