# Fingpay AEPS API Integration

This PHP library provides integration with Fingpay AEPS (Aadhaar Enabled Payment System) services.

## Features

- **Cash Withdrawal**: Withdraw cash using Aadhaar authentication
- **Balance Inquiry**: Check account balance
- **Mini Statement**: Get recent transaction statement
- **Cash Deposit**: Deposit cash to account

## Requirements

- PHP 7.0 or higher
- OpenSSL extension enabled
- cURL extension enabled

## Installation

1. Copy all files to your web server directory
2. Ensure you have the Fingpay public certificate file (`.cer` or `.txt` format)
3. Update credentials in `api_handler.php`:
   - `$MERCHANT_ID` - Your Fingpay merchant ID
   - `$TERMINAL_ID` - Your terminal ID
   - `$DEFAULT_DEVICE_IMEI` - Default device IMEI (optional)
4. Ensure PHP has OpenSSL and cURL extensions enabled

## Web Interface

A complete HTML web interface is provided for easy transaction processing:

1. Open `index.html` in your web browser
2. Use the tabs to switch between different transaction types:
   - **Cash Withdrawal**: Withdraw cash from account
   - **Balance Inquiry**: Check account balance
   - **Mini Statement**: View recent transactions
   - **Cash Deposit**: Deposit cash to account
3. Fill in the required fields and submit
4. View the response in real-time

The interface uses AJAX to communicate with `api_handler.php` which processes all transactions with hardcoded credentials.

## Usage

### Basic Setup

```php
require_once 'FingpayAEPS.php';

// Path to the public certificate file
$certPath = 'path/to/fingpay_public_production.cer';

// Initialize the client
$fingpay = new FingpayAEPS($certPath, 30); // timeout in seconds
```

### Cash Withdrawal

```php
$response = $fingpay->cashWithdrawal(
    'MERCHANT_ID',           // Your merchant ID
    'TERMINAL_ID',           // Your terminal ID
    '123456789012',          // Customer Aadhaar number
    1000.00,                 // Amount to withdraw
    'John Doe',              // Customer name
    'TXN' . time(),          // Unique transaction reference
    'BANK_CODE',             // Bank code
    'DEVICE_IMEI',           // Device IMEI (optional)
    []                       // Additional parameters (optional)
);

print_r($response);
```

### Balance Inquiry

```php
$response = $fingpay->balanceInquiry(
    'MERCHANT_ID',
    'TERMINAL_ID',
    '123456789012',          // Aadhaar number
    'BANK_CODE',
    'TXN' . time(),          // Unique transaction reference
    'DEVICE_IMEI'            // Device IMEI (optional)
);

print_r($response);
```

### Mini Statement

```php
$response = $fingpay->miniStatement(
    'MERCHANT_ID',
    'TERMINAL_ID',
    '123456789012',          // Aadhaar number
    'BANK_CODE',
    'TXN' . time(),          // Unique transaction reference
    'DEVICE_IMEI'            // Device IMEI (optional)
);

print_r($response);
```

### Cash Deposit

```php
$response = $fingpay->cashDeposit(
    'MERCHANT_ID',
    'TERMINAL_ID',
    '123456789012',          // Aadhaar number
    500.00,                  // Amount to deposit
    'Jane Doe',              // Customer name
    'TXN' . time(),          // Unique transaction reference
    'BANK_CODE',
    'DEVICE_IMEI',           // Device IMEI (optional)
    []                       // Additional parameters (optional)
);

print_r($response);
```

## Security

- All requests are encrypted using AES-128-CBC encryption
- The AES session key is encrypted using RSA public key encryption
- Each request includes a SHA256 hash for integrity verification
- SSL/TLS verification is enabled by default

## Error Handling

The library includes comprehensive error handling:

```php
try {
    $response = $fingpay->cashWithdrawal(...);
    
    if (isset($response['status']) && $response['status'] === 'error') {
        echo "Error: " . $response['message'];
    } else {
        // Process successful response
    }
} catch (Exception $e) {
    echo "Exception: " . $e->getMessage();
}
```

## Response Format

The API returns responses in JSON format. Example successful response:

```json
{
    "status": "success",
    "transactionId": "TXN123456",
    "message": "Transaction successful",
    "data": { ... }
}
```

Example error response:

```json
{
    "status": "error",
    "message": "Invalid Aadhaar number",
    "error": "AADHAAR_INVALID"
}
```

## Notes

- Always use unique transaction references for each request
- Ensure the certificate file path is correct and accessible
- Keep your merchant credentials secure
- The device IMEI is optional but recommended for transaction tracking
- Check the Fingpay API documentation for required field formats and values

## Files Structure

- `FingpayAEPS.php` - Main AEPS API class
- `api_handler.php` - PHP handler for web interface (contains hardcoded credentials)
- `index.html` - Web interface for transactions
- `example_usage.php` - Example PHP code for direct API usage
- `bank_list.php` - Bank list with common AEPS bank codes
- `get_bank_list.php` - API endpoint to fetch bank list
- `BANK_CODES_INFO.md` - Detailed information about bank codes

## Web Interface Setup

1. **Configure Credentials**: Edit `api_handler.php` and update:
   ```php
   $MERCHANT_ID = 'YOUR_MERCHANT_ID';
   $TERMINAL_ID = 'YOUR_TERMINAL_ID';
   $DEFAULT_DEVICE_IMEI = 'DEFAULT_IMEI_12345';
   ```

2. **Certificate Path**: Ensure the certificate file path is correct in `api_handler.php`:
   ```php
   $certPath = __DIR__ . '/../fingpay_public_production.txt';
   ```

3. **Access the Interface**: Open `index.html` in a web browser or access via your web server.

## Bank Codes

Bank codes are required for all AEPS transactions. The web interface includes a dropdown with common bank codes for easy selection.

**Bank Code Sources:**
- **Built-in List**: Common bank codes are provided in `bank_list.php`
- **Web Interface**: The HTML interface automatically loads banks in dropdown menus
- **API Endpoint**: Access bank list via `get_bank_list.php` endpoint

**To get bank codes:**
1. Use the dropdown in the web interface (automatically populated)
2. Check `bank_list.php` for the complete list
3. Access `get_bank_list.php` endpoint programmatically
4. Refer to `BANK_CODES_INFO.md` for detailed information

**Note:** If Fingpay provides a bank list API in their documentation, you can implement it in `bank_list.php`. See `BANK_CODES_INFO.md` for implementation details.

## Support

For API-related queries, refer to the official Fingpay API documentation.

