# Bank Codes for AEPS Transactions

## Overview

Bank codes are unique identifiers used in AEPS (Aadhaar Enabled Payment System) transactions to specify which bank the customer's account belongs to. These are typically 6-digit numeric codes standardized by NPCI (National Payments Corporation of India).

## How to Get Bank Codes

### Option 1: Use the Built-in Bank List (Recommended)

The integration includes a `bank_list.php` file containing common AEPS bank codes. The web interface (`index.html`) automatically loads these banks in dropdown menus for easy selection.

**Access the bank list:**
- Open `index.html` in your browser
- The bank dropdown will be populated automatically with available banks
- Select the appropriate bank from the dropdown

### Option 2: Fingpay Bank List API (If Available)

If Fingpay provides a bank list API endpoint in their documentation, you can implement it in `bank_list.php` using the `fetchBankListFromAPI()` function.

**Typical API endpoint might be:**
```
GET /fpaepsservice/api/bankList
```

**Check the Fingpay API documentation for:**
- Bank list API endpoint URL
- Required authentication/headers
- Request/response format

### Option 3: Static Bank List

A static list of common bank codes is provided in `bank_list.php`. The list includes major banks like:

- **607094** - State Bank of India (SBI)
- **607027** - Punjab National Bank (PNB)
- **606985** - Bank of Baroda (BOB)
- **607057** - ICICI Bank
- **607010** - HDFC Bank
- **607034** - Axis Bank
- And many more...

## Common Bank Codes Reference

| Bank Code | Bank Name |
|-----------|-----------|
| 607094 | State Bank of India (SBI) |
| 607027 | Punjab National Bank (PNB) |
| 606985 | Bank of Baroda (BOB) |
| 607057 | ICICI Bank |
| 607010 | HDFC Bank |
| 607034 | Axis Bank |
| 607005 | Canara Bank |
| 607003 | Union Bank of India |
| 607024 | Indian Bank |
| 607013 | IDBI Bank |
| 607039 | Bank of India |
| 607044 | Yes Bank |
| 607043 | Kotak Mahindra Bank |
| 607045 | IndusInd Bank |

*Note: This is not an exhaustive list. Refer to NPCI or Fingpay documentation for the complete list.*

## API Endpoint to Get Bank List

The integration provides an endpoint to retrieve the bank list programmatically:

**Endpoint:** `get_bank_list.php`

**Request:**
```http
GET /get_bank_list.php
```

**Response:**
```json
{
    "status": "success",
    "banks": [
        {
            "code": "607094",
            "name": "State Bank of India (SBI)"
        },
        {
            "code": "607027",
            "name": "Punjab National Bank (PNB)"
        }
        // ... more banks
    ]
}
```

## Usage in Code

### PHP Example:

```php
require_once 'bank_list.php';

// Get all banks
$banks = getBankList();

// Get bank name by code
$bankName = getBankName('607094'); // Returns "State Bank of India (SBI)"
```

### JavaScript Example (Web Interface):

The bank list is automatically loaded when you open `index.html`. You can also fetch it manually:

```javascript
fetch('get_bank_list.php')
    .then(response => response.json())
    .then(data => {
        if (data.status === 'success') {
            console.log(data.banks); // Array of banks with code and name
        }
    });
```

## Important Notes

1. **Bank Code Format**: Bank codes are typically 6-digit numeric strings
2. **Case Sensitivity**: Bank codes are case-sensitive (always numeric)
3. **Validation**: Always validate bank codes before sending API requests
4. **Updates**: Bank codes may change or new banks may be added. Keep your bank list updated
5. **Documentation**: Refer to Fingpay API documentation for the most accurate and up-to-date bank codes

## If Fingpay Provides Bank List API

If the Fingpay API documentation mentions a bank list endpoint, follow these steps:

1. **Check the API Documentation**: Look for endpoints like:
   - `/api/bankList`
   - `/api/getBanks`
   - `/api/supportedBanks`

2. **Note the Requirements**:
   - Authentication method
   - Required headers
   - Request format
   - Response format

3. **Update `bank_list.php`**: Modify the `fetchBankListFromAPI()` function to implement the actual API call

4. **Example Implementation**:
```php
function fetchBankListFromAPI($certPath) {
    // Initialize Fingpay client (if needed for authentication)
    $fingpay = new FingpayAEPS($certPath);
    
    // Make API request to bank list endpoint
    $response = $fingpay->makeRequest('/fpaepsservice/api/bankList', []);
    
    // Parse and return bank list
    // Format: [['code' => '607094', 'name' => 'State Bank of India']]
    return $response;
}
```

## Support

For bank code related queries:
1. Check Fingpay API documentation
2. Contact Fingpay support
3. Refer to NPCI documentation for AEPS standards

