# API Migration Guide - No Hardcoded Values

This guide explains how the system has been updated to use API endpoints from Fingpay API documentation instead of hardcoded values.

## Changes Made

### 1. API Configuration File (`api_config.php`)
- **Created**: Centralized configuration file for all API endpoints
- **Location**: `code/api_config.php`
- **Purpose**: Contains all API endpoints, base URL, and configuration settings
- **Benefits**: Easy to update endpoints without modifying core code

### 2. Updated FingpayAEPS Class (`FingpayAEPS.php`)
- **Removed**: Hardcoded endpoints and base URL constants
- **Added**: 
  - Dynamic configuration loading from `api_config.php`
  - Support for custom configuration
  - `getBankList()` method to fetch banks from API
  - `makeSimpleRequest()` method for non-encrypted API calls (like bank list)
- **Benefits**: Flexible endpoint management, support for API-based bank list

### 3. Updated Bank List System (`bank_list.php`)
- **Removed**: Hardcoded bank list as primary source
- **Added**: 
  - `fetchBankListFromAPI()` - Fetches banks from Fingpay API
  - Automatic fallback to static list if API fails
  - Support for multiple response formats from API
- **Benefits**: Always up-to-date bank list from API, graceful fallback

### 4. Updated Bank List Endpoint (`get_bank_list.php`)
- **Changed**: Now fetches from API first, falls back to static list
- **Added**: 
  - API integration
  - Response metadata (source, count)
  - Automatic sorting by bank name
- **Benefits**: Real-time bank data from Fingpay API

## Configuration

### API Endpoints Configuration

Edit `code/api_config.php` to update API endpoints based on Fingpay documentation:

```php
return [
    'base_url' => 'https://fingpayap.tapits.in',
    
    'endpoints' => [
        'cash_withdrawal' => '/fpaepsservice/api/cashWithdrawal/merchant/withdrawal',
        'balance_inquiry' => '/fpaepsservice/api/balanceInquiry/merchant/inquiry',
        'mini_statement' => '/fpaepsservice/api/miniStatement/merchant/statement',
        'cash_deposit' => '/fpaepsservice/api/cashDeposit/merchant/deposit',
        
        // Update this endpoint based on actual Fingpay API documentation
        'bank_list' => '/fpaepsservice/api/bankList',
    ],
    
    'bank_list_config' => [
        'requires_encryption' => false, // Set to true if bank list API requires encryption
        'method' => 'GET', // or 'POST'
    ],
];
```

### Bank List API Endpoint

**Important**: The bank list endpoint may vary. Update the `bank_list` endpoint in `api_config.php` based on your Fingpay API documentation. Common patterns:

- `/fpaepsservice/api/bankList`
- `/fpaepsservice/api/bankList/merchant/list`
- `/fpaepsservice/api/getBankList`
- `/fpaepsservice/api/supportedBanks`

### Bank List Configuration

If the bank list API requires encryption or uses POST method, update `bank_list_config` in `api_config.php`:

```php
'bank_list_config' => [
    'requires_encryption' => true,  // Set to true if encryption needed
    'method' => 'POST',             // Change if API uses POST
],
```

## How It Works

### 1. Bank List Loading Flow

```
User Request → get_bank_list.php
    ↓
getBankList() function
    ↓
Try Fingpay API (fetchBankListFromAPI)
    ↓
API Success? → Use API data
    ↓
API Failed? → Use static fallback list
    ↓
Return to frontend
```

### 2. API Response Format Handling

The system automatically handles multiple response formats:

1. **Array format**: `[{"code": "607094", "name": "SBI"}, ...]`
2. **Banks key format**: `{"banks": [{"code": "...", "name": "..."}]}`
3. **Data key format**: `{"data": [{"code": "...", "name": "..."}]}`
4. **Object format**: `{"607094": "SBI", ...}`

### 3. Fallback Mechanism

If the API call fails (network error, API unavailable, wrong endpoint), the system automatically falls back to the static bank list to ensure the application continues working.

## Testing

### Test Bank List API

1. **Enable API mode**: Ensure `$useAPI = true` in `get_bank_list.php`
2. **Update endpoint**: Verify the `bank_list` endpoint in `api_config.php` matches Fingpay documentation
3. **Check response**: Open browser console and check `get_bank_list.php` response
4. **Verify fallback**: Temporarily break the endpoint to verify fallback works

### Verify Configuration

Check that all endpoints in `api_config.php` match your Fingpay API documentation:
- Base URL
- Endpoint paths
- Bank list endpoint
- Encryption requirements

## API Documentation Reference

When you receive the official Fingpay API documentation:

1. **Bank List Endpoint**: Look for endpoints like:
   - Bank list
   - Get banks
   - Supported banks
   - Aadhaar banks

2. **Request Format**: Note:
   - HTTP method (GET/POST)
   - Required parameters
   - Authentication requirements
   - Encryption requirements

3. **Response Format**: Document:
   - JSON structure
   - Field names (code, name, etc.)
   - Error formats

4. **Update Configuration**: Modify `api_config.php` accordingly:
   - Update endpoint path
   - Set encryption flag
   - Set HTTP method
   - Update response parser if needed

## Troubleshooting

### Bank List Not Loading from API

1. **Check endpoint**: Verify `bank_list` endpoint in `api_config.php`
2. **Check encryption**: Ensure `requires_encryption` matches API requirements
3. **Check method**: Verify HTTP method (GET/POST) is correct
4. **Check certificate**: Ensure certificate path is correct
5. **Check response format**: Review API response and update parser if needed

### API Returns Error

1. **Check network**: Verify internet connection
2. **Check certificate**: Ensure certificate file exists and is valid
3. **Check credentials**: Verify merchant ID if required
4. **Check endpoint**: Confirm endpoint URL is correct
5. **Review logs**: Check PHP error logs for details

### Fallback Not Working

1. **Verify static list**: Ensure `$FALLBACK_BANK_CODES` is defined in `bank_list.php`
2. **Check error handling**: Review exception handling in `fetchBankListFromAPI()`

## Benefits

✅ **No Hardcoded Values**: All endpoints configurable via `api_config.php`  
✅ **API-Driven**: Bank list fetched from Fingpay API  
✅ **Graceful Degradation**: Automatic fallback if API fails  
✅ **Flexible Configuration**: Easy to update endpoints  
✅ **Future-Proof**: Easy to add new endpoints  
✅ **Maintainable**: Centralized configuration  

## Next Steps

1. Review Fingpay API documentation
2. Update `api_config.php` with correct endpoints
3. Test bank list API integration
4. Verify all endpoints work correctly
5. Remove fallback static list if desired (after confirming API stability)

